package com.xunlei.netty.httpserver;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.lang.management.ManagementFactory;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.URL;
import java.util.concurrent.ExecutorService;

import javax.annotation.Resource;

import org.jboss.netty.bootstrap.ServerBootstrap;
import org.jboss.netty.channel.socket.nio.NioServerSocketChannelFactory;
import org.jboss.netty.logging.InternalLoggerFactory;
import org.jboss.netty.logging.Slf4JLoggerFactory;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import com.xunlei.netty.httpserver.component.AbstractPageDispatcher;
import com.xunlei.netty.httpserver.util.CloseableHelper;
import com.xunlei.netty.httpserver.util.HttpServerConfig;
import com.xunlei.netty.httpserver.util.IPAuthenticator;
import com.xunlei.netty.httpserver.util.Log;
import com.xunlei.netty.httpserver.util.NetUtil;
import com.xunlei.netty.httpserver.util.NioSocketBootstrapFactory;
import com.xunlei.netty.httpserver.util.spring.BeanUtil;
import com.xunlei.netty.httpserver.util.spring.Config;
import com.xunlei.netty.httpserver.util.spring.SpringBootstrap;

/**
 * @author wangcanyi
 */
@Component
public class Bootstrap {

    public static ApplicationContext CONTEXT;
    private static final Logger log = Log.getLogger();
    private volatile boolean stopping = false;

    /**
     * args里是否含有参数
     * @param args
     * @param what
     * @return
     */
    public static boolean isArgWhat(String[] args, String... what) {
        if (args == null || args.length == 0) {
            return false;
        }
        String arg = args[0].toLowerCase();
        for (String w : what) {
            if (arg.indexOf(w) >= 0) {
                return true;
            }
        }
        return false;
    }

    public static ApplicationContext main(String[] args, Runnable initialRunnable, Runnable shutdownRunnalbe, String... springConfigLocations) throws IOException {
    	log.debug("Netty服务.服务Main运行");
    	long before = System.currentTimeMillis();
        try {
            CONTEXT = SpringBootstrap.load(springConfigLocations);
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1); // 遇到任何异常直接退出
        }
        System.err.println("----->loadContext      [" + (System.currentTimeMillis() - before) + "MS]");
        Bootstrap bootstrap = BeanUtil.getTypedBean("bootstrap");
        bootstrap.shutdownRunanble = shutdownRunnalbe;
        if (isArgWhat(args, "stop", "shutdown")) {
        	log.debug("Netty服务.服务关闭");
            bootstrap.sendShutdownCmd();
            System.exit(0);
        } else {
        	log.debug("Netty服务.服务启动");
            // 2012-05-29
            // Ruiz增加启动参数compelled 在发现需要监听的http端口已经被占用时，会尝试关闭前个Ruiz再来监听此端口，
            // compelled用于 替代原来老的 restart 参数,这种方式 更趋近于 无缝重启（中间重启的间隔更短）
            if (!isArgWhat(args, "compelled", "force")) {
                bootstrap.bindRetryTimeout = 0;
            }
            bootstrap.start(initialRunnable);
        }
        return CONTEXT;
    }

    public static ApplicationContext main(String[] args, Runnable initialRunnable, String... springConfigLocations) throws IOException {
        return main(args, initialRunnable, null, springConfigLocations);
    }

    public static ApplicationContext main(String[] args, String... springConfigLocations) throws IOException {
        return main(args, null, null, springConfigLocations);
    }

    private ExecutorService bossExecutor = HttpServerConfig.bossExecutor;
    private Runnable shutdownRunanble = null;
    @Resource
    private HttpServerPipelineFactory httpServerPipelineFactory;
    @Autowired
    private HttpServerConfig config;
    @Autowired
    private AbstractPageDispatcher pageDispatcher;
    private NioServerSocketChannelFactory serverSocketChannelFactory;
    private ExecutorService workerExecutor = HttpServerConfig.workerExecutor;
    @Config
    private long bindRetryTimeout = 60000;// 绑定重试的超时时间，默认60s

    /**
     * 初始化环境
     */
    private void initEnv() {
        InternalLoggerFactory.setDefaultFactory(new Slf4JLoggerFactory());
        IPAuthenticator.setBusinessWhiteIp(config.getIpWhiteList());//设置Ip白名单
        IPAuthenticator.reload();//加载配置
        // 2012-05-29 判断端口是否已经被占用，挪到真正监听时判断
        // int https_port = config.getHttps_listen_port();
        // int port = config.getListen_port();
        // NetUtil.checkSocketPortBind(port, https_port);
    }

    /**
     * 初始化外部逻辑
     * @param initialRunnable
     */
    private void initOutter(Runnable initialRunnable) {
        if (initialRunnable != null) {
            long before = System.currentTimeMillis();
            try {
                initialRunnable.run();
            } catch (Throwable t) { // 启动时如果有异常就自动退出
                t.printStackTrace();
                System.exit(-1);
            }
            System.err.println("----->initialRunnable  [" + (System.currentTimeMillis() - before) + "MS]");
        }
    }

    private ServerBootstrap initServerBootstrap(HttpServerPipelineFactory channelPipelineFactory) {
        ServerBootstrap serverBootstrap = new ServerBootstrap(serverSocketChannelFactory);
        serverBootstrap.setPipelineFactory(channelPipelineFactory.rebuildPipeline());// Set up the event pipeline factory.
        NioSocketBootstrapFactory.setBootstrapOptions(serverBootstrap, config.getConnectTimeoutMillis(), config.getReceiveBufferSize(), config.getSendBufferSize());
        return serverBootstrap;
    }

    private void bind(int port, ServerBootstrap serverBootstrap) {
        if (port > 0) {
            // ServerBootstrap serverBootstrap = new ServerBootstrap(serverSocketChannelFactory);
            // serverBootstrap.setPipelineFactory(channelPipelineFactory.rebuildPipeline());// Set up the event pipeline factory.
            // NioSocketBootstrapFactory.setBootstrapOptions(serverBootstrap, config.getConnectTimeoutMillis(), config.getReceiveBufferSize(), config.getSendBufferSize());
            serverBootstrap.bind(new InetSocketAddress(port));// Bind and start to accept incoming connections.
        }
    }

    /**
     * 初始化服务
     */
    private void initServer() {
        // workercount 数目不能太多,因为其内部一直在 循环接收 task,占用cpu
        serverSocketChannelFactory = new NioServerSocketChannelFactory(bossExecutor, workerExecutor, config.getRealWorkerCount());
        int port = config.getListen_port();
        String ports = port + "";
        Exception ex = null;
        try {
            ServerBootstrap serverBootstrap = initServerBootstrap(httpServerPipelineFactory);
            long beginBind = System.currentTimeMillis();
            NetUtil.checkSocketPortBind(port);
            bind(port, serverBootstrap);
            long endBind = System.currentTimeMillis();
            System.err.println("----->bindHttpPort     [" + (endBind - beginBind) + "MS]");
        } catch (Exception e) {
            ex = e;
        }
        long span = System.currentTimeMillis() - ManagementFactory.getRuntimeMXBean().getStartTime();
        String msg = ex == null ? "OK" : "ERROR";
        String chnmsg = ex == null ? "服务器启动完毕.(port[" + ports + "])" : "服务器启动失败.(port[" + config.getListen_port() + "])";
        String spanStr = "[" + span + "MS]";
        String errStr = ex == null ? "" : ex.getMessage();
        HttpServerConfig.ALARMLOG.error("HTTPServer(port[{}],workerCount[{}]) Start {}.{}", new Object[] {
            ports,
            config.getRealWorkerCount(),
            msg,
            spanStr,
            ex
        });
        System.err.println(chnmsg + spanStr + errStr);
		log.debug("Netty服务.初始化服务." + chnmsg + spanStr + errStr);
        if (ex != null) {
            System.exit(1);
        }
    }

    /**
     * 服务启动
     * @param initialRunnable
     * @throws IOException
     */
    public void start(Runnable initialRunnable) throws IOException {
    	log.debug("Netty服务.初始化环境.initEnv()");
        initEnv();
    	log.debug("Netty服务.初始化业务接口.pageDispatcher.init()");
        try {
            pageDispatcher.init();//初始化业务接口
        } catch (Exception e) { // 启动时如果有异常就自动退出
        	log.error("Netty服务.初始化业务接口.异常：" + e.getMessage());
            e.printStackTrace();
            System.exit(-1);
        }
    	log.debug("Netty服务.初始化外部逻辑.initOutter()");
        initOutter(initialRunnable);
    	log.debug("Netty服务.初始化服务.initServer");
        initServer();
    }

    public long sendShutdownCmd() {
        HttpURLConnection urlConnection = null;
        LineNumberReader lineReader = null;
        try {
            URL url = new URL("http://localhost:" + config.getListen_port() + "/shutdown");
            urlConnection = (HttpURLConnection) url.openConnection();
            urlConnection.setReadTimeout(10000);
            lineReader = new LineNumberReader(new InputStreamReader(urlConnection.getInputStream()));
            StringBuilder sb = new StringBuilder();
            while (true) {
                String tmp = lineReader.readLine();
                if (tmp == null) {
                    break;
                }
                sb.append(tmp);
            }
            String returnStr = sb.toString();
            log.error("shutdown last result:{}", returnStr);
            return Long.valueOf(returnStr);
        } catch (Exception e) {
            log.error("", e);
        } finally {
            if (urlConnection != null) {
                urlConnection.disconnect();
                urlConnection = null;
            }
            CloseableHelper.closeSilently(lineReader);
        }
        return 0;
    }

    public void stop() {
        stopping = true;
        String msg = null;
        httpServerPipelineFactory.rebuildPipeline();
        if (shutdownRunanble != null) {// 再作业务关闭处理
            long before = System.currentTimeMillis();
            msg = "shutdownRunanble run begin...";
            log.error(msg);
            System.err.println(msg);
            shutdownRunanble.run();
            msg = "shutdownRunanble run end,USING " + (System.currentTimeMillis() - before) + " MS";
            log.error(msg);
            System.err.println(msg);
        }

        // 正式环境发现releaseExternalResources要花很久时间
        // 2011-02-16 因此默认不处理这些线程池
        // 如果要release,请在shudown Runnnable 中加上 HttpServerConfig.releaseExternalResources()方法来实现
        // serverBootstrap.releaseExternalResources(); // 先关httpServer
    }

    public boolean isStop() {
        return stopping;
    }
}
