package com.xunlei.netty.httpserver.component;

import java.lang.reflect.Method;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.xunlei.netty.httpserver.cmd.BaseCmd;
import com.xunlei.netty.httpserver.cmd.CmdMappers.CmdMeta;
import com.xunlei.netty.httpserver.exception.ResourceNotFoundError;
import com.xunlei.netty.httpserver.handler.TextResponseHandlerManager;
import com.xunlei.netty.httpserver.util.IPAuthenticator;
import com.xunlei.netty.httpserver.util.IPGetterHelper;
import com.xunlei.netty.httpserver.util.Log;

/**
 * 命令映射分发器
 * 
 * <pre>
 * 约定： http://host:port /cmdname [/methodname] [?callback=value&debug=1]
 * 按cmdname找到指定的cmd类(类名默认以Cmd为后缀)
 * 当指定了methodname时,调用cmd类的对应名称的方法,未指定时调用process方法
 * </pre>
 * 
 * <pre>
 * 通过CmdMapper的映射关系来找到对应命令
 * 在httpserver启动时,dispatcher会初始化cmdMappers里面的所有命令
 * </pre>
 * 
 * @author ZengDong
 * @since 2010-6-8 上午01:53:26
 */
@Service
public class CmdMapperDispatcher extends BasePageDispatcher {
	private static final Logger log = Log.getLogger();

	@Autowired
	protected TextResponseHandlerManager handlerManager;

	// private static final Logger log = Log.getLogger();

	protected Object _dispatch(XLContextAttachment attach) throws Exception {
		XLHttpRequest request = attach.getRequest();
		XLHttpResponse response = attach.getResponse();
		String path = request.getPath();
		log.debug("Netty服务.Http请求.path=" + path);
		CmdMeta meta = cmdMappers.getCmdMeta(path);
		if (meta == null) {
			throw ResourceNotFoundError.INSTANCE;
		}
		// 把attach上设置其CmdMeta
		attach.setCmdMeta(meta);

		if (meta.isDisable()) {
			throw ResourceNotFoundError.INSTANCE;
		}
		// 管理员接口，判断IP白名单
		if (meta.isAdmin()) {
			IPAuthenticator.auth(request);
		}

		BaseCmd cmd = meta.getCmd();
		Method method = meta.getMethod();
		Transaction t = Cat.newTransaction("URL", path);
		t.addData("clientIp", IPGetterHelper.getRequestIp(request));
		try {
			attach.registerProcessThread();
			cmd.processBegin(request, response);// 方法执行前，执行此方法，对业务线进行统一入口
			Object resultObj = method.invoke(cmd, request, response);
			cmd.processEnd(request, response, resultObj);// 方法执行后，执行此方法，，对业务线进行统一出口
			t.setStatus(Transaction.SUCCESS);
			return resultObj;
		} catch (Exception e) {
			t.addData("ErrorMsg", e.getMessage());
			t.setStatus(e);
			if (StringUtils.isNotBlank(e.getMessage())) {
				Cat.logError(e.getMessage(), e);// 记录异常
			} else {
				Cat.logError(e);// 记录异常
			}
			throw e;
		} finally {
			t.complete();
			attach.unregisterProcessThread();
		}
	}

	@Override
	protected void dispatch(XLContextAttachment attach) throws Exception {
		attach.checkChannelOrThread();
		Object cmdReturnObj = null;
		try {
			cmdReturnObj = _dispatch(attach);
		} catch (Throwable ex) {
			cmdReturnObj = handlerManager.handleThrowable(attach, ex);
		} finally {
			String cmdReturnStr = "Obj return Null";
			if(cmdReturnObj != null)
				cmdReturnStr = cmdReturnObj.toString();
			log.debug("Netty服务.Http请求.返回值cmdReturnObj=" + cmdReturnStr);
			handlerManager.writeResponse(attach, cmdReturnObj);
		}
	}

	@Override
	public void init() throws Exception {
		// 优先级则低到高
		log.debug("Netty服务.初始化Url映射.cmdMappers.initAutoMap()");
		cmdMappers.initAutoMap();
		log.debug("Netty服务.重置CMD配置,后台扫描器，超时功能是否启运.cmdMappers.resetCmdConfig()");
		cmdMappers.resetCmdConfig();
	}
}
