package com.xunlei.netty.httpserver.cmd.common;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import javax.annotation.Resource;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.xunlei.netty.httpserver.HttpServerPipelineFactory;
import com.xunlei.netty.httpserver.cmd.BaseStatCmd;
import com.xunlei.netty.httpserver.cmd.CmdMapper;
import com.xunlei.netty.httpserver.cmd.CmdMappers;
import com.xunlei.netty.httpserver.cmd.CmdMappers.CmdMeta;
import com.xunlei.netty.httpserver.cmd.annotation.CmdAdmin;
import com.xunlei.netty.httpserver.component.XLHttpRequest;
import com.xunlei.netty.httpserver.component.XLHttpResponse;
import com.xunlei.netty.httpserver.component.XLHttpResponse.ContentType;
import com.xunlei.netty.httpserver.util.HtmlUtil;
import com.xunlei.netty.httpserver.util.HttpServerConfig;
import com.xunlei.netty.httpserver.util.HttpUtil;
import com.xunlei.netty.httpserver.util.IPAuthenticator;
import com.xunlei.netty.httpserver.util.Log;
import com.xunlei.netty.httpserver.util.spring.ConfigBeanPostProcessor;

/**
 * 实时设置
 * 
 * @author ZengDong
 * @since 2010-5-23 上午12:15:48
 */
@SuppressWarnings("restriction")
@Service
public class SettingCmd extends BaseStatCmd {

	private static final Logger log = Log.getLogger();
	@Autowired
	private CmdMappers cmdMappers;
	@Autowired
	private HttpServerConfig config;
	@Autowired
	private ConfigBeanPostProcessor configProcessor;
	@Resource
	private HttpServerPipelineFactory httpServerPipelineFactory;

	/**
	 * <pre>
	 * 服务器的所有接口都不被爬虫收录
	 * 详情见：http://www.baidu.com/search/robots.html
	 * </pre>
	 */
	@CmdMapper("/robots.txt")
	@CmdAdmin
	public Object robots(XLHttpRequest request, XLHttpResponse response) throws Exception {
		response.setInnerContentType(ContentType.plain);
		return "User-agent: *\nDisallow: /\n";
	}

	/**
	 * <pre>
	 * setting/cmds 统计各个接口请求量数据
	 * txt 参数为true表示文本展示统计信息
	 * last 参数为true表示显示最近10分钟内的请求数据
	 * order 参数为true表示根据请求量all_num进行排序显示
	 * </pre>
	 */
	@CmdAdmin
	public Object cmds(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		boolean useTxt = request.getParameterBoolean("txt", false);
		if (!useTxt) {
			response.setInnerContentType(ContentType.html);
		}
		String tmp = "";
		Map<String, CmdMeta> cmd_urls_map = cmdMappers.getCmdAllMap();
		if(cmd_urls_map != null && cmd_urls_map.size() > 0){
			List<Map<String,String>> tableData = new ArrayList<Map<String,String>>();
			for (Entry<String, CmdMeta> cmdUrlMap : cmd_urls_map.entrySet()) {
				String cmdUrl = cmdUrlMap.getKey();
				CmdMeta cmdMeta = cmdUrlMap.getValue();
				Map<String,String> mapData = new LinkedHashMap<String, String>();
				mapData.put("url",HtmlUtil.getHtmlLink(cmdUrl));
				mapData.put("name",cmdMeta.toString());
				mapData.put("isAdmin",String.valueOf(cmdMeta.isAdmin()));
				mapData.put("timeout",String.valueOf(cmdMeta.getTimeout()));
				tableData.add(mapData);
			}
			tmp = HtmlUtil.getHtmlTable(tableData);
		}
		return tmp;
	}

	/**
	 * 显示当前config
	 */
	@CmdAdmin
	public Object config(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		return configProcessor.printCurrentConfig(new StringBuilder());
	}

	/**
	 * 显示调参的历史记录
	 */
	@CmdAdmin
	public Object configHistory(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		StringBuilder tmp = new StringBuilder();
		return tmp.append(configProcessor.getResetHistory());
	}

	/**
	 * 显示httpServer内部配置
	 */
	@CmdAdmin
	public Object httpServerConfig(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		StringBuilder tmp = new StringBuilder();
		for (Field f : config.getClass().getDeclaredFields()) {
			if (Modifier.isStatic(f.getModifiers())) {
				continue;
			}
			f.setAccessible(true);
			try {
				tmp.append(String.format("%-24s%-10s\n", f.getName() + ":", f.get(config)));
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		tmp.append("本地IP:\t\t\t").append(HttpUtil.getLocalIP()).append("\n");
		tmp.append("IP白名单:\t\t").append(IPAuthenticator.getIPWhiteList()).append("\n");
		try {
			tmp.append("PIPELINE:\t\t").append(httpServerPipelineFactory.getPipeline()).append("\n");
		} catch (Exception e) {
			log.error("", e);
			tmp.append(e.getMessage()).append("\n");
		}
		return tmp.toString();
	}

	/**
	 * 重加载config
	 */
	@CmdAdmin
	public Object reloadConfig(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		StringBuilder tmp = new StringBuilder();
		configProcessor.reloadConfig(tmp);
		return tmp.toString();
	}

	/**
	 * 重加载所有命令的设置
	 */
	@CmdAdmin
	public Object reloadCmdConfig(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		//return cmdMappers.resetCmdConfig();
		return null;
	}

	/**
	 * 重加载ipfilter
	 */
	@CmdAdmin
	public Object reloadIpfilter(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		IPAuthenticator.reload();
		return "reset success";
	}

	/**
	 * 重置受临时调参保护的config
	 */
	@CmdAdmin
	public Object resetGuardedConfig(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		StringBuilder tmp = new StringBuilder("reset guarded config...\n");
		configProcessor.resetGuradedConfig(tmp);
		return tmp;
	}

	/**
	 * 临时调参
	 */
	@CmdAdmin
	public Object setConfig(XLHttpRequest request, XLHttpResponse response) throws Exception {
		init(request, response);
		StringBuilder tmp = new StringBuilder("set tmp config...\n");
		for (Map.Entry<String, List<String>> e : request.getParameters().entrySet()) {
			String fieldName = e.getKey();
			String value = e.getValue().get(0);
			configProcessor.setFieldValue(fieldName, value, tmp);
		}
		return tmp.toString();
	}
}
