package com.xunlei.channel.common.utils.encrypt;

import java.io.*;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.util.encoders.UrlBase64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * RSA加密工具类
 * <p>
 * 生成RSA相关密钥,在安装有openssl的linux下执行以下命令,私钥己方保存,公钥对外公开,签名时使用私钥加密,公钥解密
 * <p>
 * 1.生成RSA私钥 openssl genrsa -out rsaprivatekey.pem 1024
 * <p>
 * 2.生成对应RSA公钥 openssl rsa -in rsaprivatekey.pem -pubout -out rsapublickey.pem
 * <p>
 * 3.将RSA私钥转换成PKCS8格式 openssl pkcs8 -topk8 -inform PEM -in rsaprivatekey.pem -outform PEM -nocrypt -out rsaprivatepkcs8.pem
 *
 * @author huangchunhui 2014年6月4日 上午10:36:09
 */
public class RSASignature extends RSA {

    private static final Logger LOG = LoggerFactory.getLogger(RSASignature.class);
    public static final String KEY_ALGORITHM = "RSA";
    public static final String SIGNATURE_ALGORITHM = "SHA1WithRSA";
    public static final String ENCODING = "utf-8";
    public static final String X509 = "X.509";


    /**
     * RSA私钥签名
     *
     * @param content 待签名数据
     * @param privateKey 私钥
     * @return 签名值
     */
    public static String signByPrivateKey(String content, String privateKey) {
        try {
            PrivateKey priKey = getPrivateKey(privateKey);
            Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
            signature.initSign(priKey);
            signature.update(content.getBytes(ENCODING));
            byte[] signed = signature.sign();
            return new String(UrlBase64.encode(signed), ENCODING);
        } catch (Exception e) {
            LOG.error("sign error, content: {}, priKey: {}", new Object[] { content, privateKey });
            LOG.error("sign error", e);
        }
        return null;
    }

    /**
     * RSA校验签名
     *
     * @param content 待签名数据原始串
     * @param sign 签名值
     * @param publicKey 公钥
     * @return true通过 false失败
     */
    public static boolean verifySignByPublicKey(String content, String sign, String publicKey) {
        try {
            PublicKey pubKey = getPublicKey(publicKey);

            Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
            signature.initVerify(pubKey);
            signature.update(content.getBytes(ENCODING));

            return signature.verify(UrlBase64.decode(sign.getBytes(ENCODING)));

        } catch (Exception e) {
            LOG.error("sign error, content: {}, sign: {}, pubKey: {}", new Object[] { content, sign, publicKey });
            LOG.error("sign error", e);
        }
        return false;
    }

    /**
     * RSA校验签名
     *
     * @param content 待签名数据原始串
     * @param sign 签名值
     * @param publicKey 公钥
     * @return true通过 false失败
     */
    public static boolean verifySignByPublicKey(String content, String sign, String publicKey,String encoding) {
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            byte[] encodedKey = UrlBase64.decode(publicKey.getBytes(encoding));
            PublicKey pubKey = keyFactory.generatePublic(new X509EncodedKeySpec(encodedKey));

            Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
            signature.initVerify(pubKey);
            signature.update(content.getBytes(encoding));

            return signature.verify(UrlBase64.decode(sign.getBytes(encoding)));

        } catch (Exception e) {
            LOG.error("sign error, content: {}, sign: {}, pubKey: {}", new Object[] { content, sign, publicKey });
            LOG.error("sign error", e);
        }
        return false;
    }

}
