package com.xunlei.channel.common.utils.encrypt;

import java.io.*;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.util.encoders.UrlBase64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * RSA加密工具类
 * <p>
 * 生成RSA相关密钥,在安装有openssl的linux下执行以下命令,私钥己方保存,公钥对外公开,签名时使用私钥加密,公钥解密
 * <p>
 * 1.生成RSA私钥 openssl genrsa -out rsaprivatekey.pem 1024
 * <p>
 * 2.生成对应RSA公钥 openssl rsa -in rsaprivatekey.pem -pubout -out rsapublickey.pem
 * <p>
 * 3.将RSA私钥转换成PKCS8格式 openssl pkcs8 -topk8 -inform PEM -in rsaprivatekey.pem -outform PEM -nocrypt -out rsaprivatepkcs8.pem
 *
 * @author huangchunhui 2014年6月4日 上午10:36:09
 */
public class RSASignature {

    private static final Logger LOG = LoggerFactory.getLogger(RSASignature.class);
    public static final String KEY_ALGORITHM = "RSA";
    public static final String SIGNATURE_ALGORITHM = "SHA1WithRSA";
    public static final String ENCODING = "utf-8";
    public static final String X509 = "X.509";

    /**
     * 获取私钥
     *
     * @param key
     * @return 返回私钥
     * @throws Exception
     */
    public static PrivateKey getPrivateKey(String key) throws Exception {
//        byte[] keyBytes = UrlBase64.decode(key.getBytes(ENCODING));
        byte[] keyBytes = buildPKCS8Key(key);
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        PrivateKey privateKey = keyFactory.generatePrivate(keySpec);
        return privateKey;
    }

    /**
     * 获取公钥
     *
     * @param key
     * @return 返回公钥
     * @throws Exception
     */
    public static PublicKey getPublicKey(String key) throws Exception {
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        key = getKey(key);
        byte[] encodedKey = UrlBase64.decode(key.getBytes(ENCODING));
        PublicKey pubKey = keyFactory.generatePublic(new X509EncodedKeySpec(encodedKey));
        return pubKey;
    }

    /**
     * RSA私钥签名
     *
     * @param content 待签名数据
     * @param privateKey 私钥
     * @return 签名值
     */
    public static String signByPrivateKey(String content, String privateKey) {
        try {
            PrivateKey priKey = getPrivateKey(privateKey);
            Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
            signature.initSign(priKey);
            signature.update(content.getBytes(ENCODING));
            byte[] signed = signature.sign();
            return new String(UrlBase64.encode(signed), ENCODING);
        } catch (Exception e) {
            LOG.error("sign error, content: {}, priKey: {}", new Object[] { content, privateKey });
            LOG.error("sign error", e);
        }
        return null;
    }

    /**
     * RSA校验签名
     *
     * @param content 待签名数据原始串
     * @param sign 签名值
     * @param publicKey 公钥
     * @return true通过 false失败
     */
    public static boolean verifySignByPublicKey(String content, String sign, String publicKey) {
        try {
            PublicKey pubKey = getPublicKey(publicKey);

            Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
            signature.initVerify(pubKey);
            signature.update(content.getBytes(ENCODING));

            return signature.verify(UrlBase64.decode(sign.getBytes(ENCODING)));

        } catch (Exception e) {
            LOG.error("sign error, content: {}, sign: {}, pubKey: {}", new Object[] { content, sign, publicKey });
            LOG.error("sign error", e);
        }
        return false;
    }

    /**
     * RSA校验签名
     *
     * @param content 待签名数据原始串
     * @param sign 签名值
     * @param publicKey 公钥
     * @return true通过 false失败
     */
    public static boolean verifySignByPublicKey(String content, String sign, String publicKey,String ENCODING) {
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            byte[] encodedKey = UrlBase64.decode(publicKey.getBytes(ENCODING));
            PublicKey pubKey = keyFactory.generatePublic(new X509EncodedKeySpec(encodedKey));

            Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
            signature.initVerify(pubKey);
            signature.update(content.getBytes(ENCODING));

            return signature.verify(UrlBase64.decode(sign.getBytes(ENCODING)));

        } catch (Exception e) {
            LOG.error("sign error, content: {}, sign: {}, pubKey: {}", new Object[] { content, sign, publicKey });
            LOG.error("sign error", e);
        }
        return false;
    }

    private static byte[] buildPKCS8Key(String privateKey) throws IOException {
        if (privateKey.contains("-----BEGIN PRIVATE KEY-----")) {
            return Base64.decodeBase64(privateKey.replaceAll("-----\\w+ PRIVATE KEY-----", ""));
        } else if (privateKey.contains("-----BEGIN RSA PRIVATE KEY-----")) {
            final byte[] innerKey = Base64.decodeBase64(privateKey.replaceAll("-----\\w+ RSA PRIVATE KEY-----", ""));
            final byte[] result = new byte[innerKey.length + 26];
            System.arraycopy(Base64.decodeBase64("MIIEvAIBADANBgkqhkiG9w0BAQEFAASCBKY="), 0, result, 0, 26);
            System.arraycopy(BigInteger.valueOf(result.length - 4).toByteArray(), 0, result, 2, 2);
            System.arraycopy(BigInteger.valueOf(innerKey.length).toByteArray(), 0, result, 24, 2);
            System.arraycopy(innerKey, 0, result, 26, innerKey.length);
            return result;
        } else {
            return Base64.decodeBase64(privateKey);
        }
    }

    public static String getKey(String content) throws Exception {
        return content.replaceAll("\\-{5}[\\w\\s]+\\-{5}[\\r\\n|\\n]", "");
    }

    /**
     * 从文件中读取密钥字符串
     * <p>
     * unSafe
     *
     * @param path 密钥绝对路径
     * @return
     * @throws Exception
     */
    public static String readSourceKey(String path) {
        BufferedReader reader = null;
        StringBuffer sourceKey = new StringBuffer();
        try {
            reader = new BufferedReader(new FileReader(path));
            String part = reader.readLine();// 按行读取
            part = reader.readLine();
            // 跳过----注释的部分
            while (part.charAt(0) != '-') {
                sourceKey.append(part + "\r");
                part = reader.readLine();
            }
        } catch (Exception e) {
            LOG.error("readSourceKey throws Exception:", e);
        } finally {
            try {
                if (null != reader) {
                    reader.close();
                }
            } catch (Exception e) {
                LOG.error("", e);
            }
        }
        return sourceKey.toString();
    }
}
