package com.xunlei.channel.common.utils.http;

import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

/**
 * @author xiongyingqi
 * @since 16-12-17 下午6:55
 */
public abstract class ParameteredHttpUtils extends HttpUtils {
    private static final Logger logger = LoggerFactory.getLogger(ParameteredHttpUtils.class);
    private static Map<Class<?>, HttpParameterHandler<?>> httpParameterHandlerMap = new WeakHashMap<Class<?>,
            HttpParameterHandler<?>>();

    public static String doPost(Object parameterEntity, String url, String charset, int timeout) throws UnsupportedEncodingException {
        if (parameterEntity == null) {
            return null;
        }
        Class<?> clazz = parameterEntity.getClass();
        HttpParameterHandler<?> httpParameterHandler = httpParameterHandlerMap.get(clazz);
        if (httpParameterHandler == null) {
            synchronized (clazz) {
                httpParameterHandler = new HttpParameterHandler(clazz);
                httpParameterHandlerMap.put(clazz, httpParameterHandler);
            }
        }
        if(logger.isDebugEnabled()) {
            logger.debug("Use HttpParameterHandler to parse entity: {}", httpParameterHandler);
        }
        String response = doPost(httpParameterHandler, parameterEntity, url, charset, timeout);
        if(logger.isDebugEnabled()) {
            logger.debug("Got response: {} by url: {}, entity: {}", new Object[]{response, url, parameterEntity});
        }
        return response;
    }

    public static String doPost(Map<String, String> parameterMap, String url, String charset, int timeout) throws
        UnsupportedEncodingException {
        if (parameterMap == null) {
            return null;
        }
        List<NameValuePair> pairs = parseMapToPairs(parameterMap);
        HttpEntity entity = new UrlEncodedFormEntity(pairs, charset);
        String response = post(url, entity, charset, timeout);
        if(logger.isDebugEnabled()) {
            logger.debug("Got response: {} by url: {}, entity: {}", new Object[]{response, url, entity});
        }
        return response;
    }

    public static String doPost(HttpParameterHandler<?> handler, Object instance, String url, String charset,
                                int timeout) throws UnsupportedEncodingException {
        Map<String, String> parameterMap = handler.getParameterMapWithObject(instance);
        if(logger.isDebugEnabled()) {
            logger.debug("Convert entity: {} to parameters: {}", instance, parameterMap);
        }
        List<NameValuePair> pairs = parseMapToPairs(parameterMap);
        HttpEntity entity = new UrlEncodedFormEntity(pairs, charset);
        return post(url, entity, charset, timeout);
    }

    public static <T> String doPostWithParameterHandler(HttpParameterHandler<T> handler, T instance, String url, String
            charset, int timeout) throws UnsupportedEncodingException {
        Map<String, String> parameterMap = handler.getParameterMap(instance);
        List<NameValuePair> pairs = parseMapToPairs(parameterMap);
        HttpEntity entity = new UrlEncodedFormEntity(pairs, charset);
        return post(url, entity, charset, timeout);
    }

}
