package com.xunlei.channel.common.utils.carriers;

import com.google.common.base.Optional;
import com.google.common.base.Strings;
import com.xunlei.channel.common.utils.carriers.config.MobileCarriersPropertiesConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author xiongyingqi
 * @since 16-12-2 下午2:41
 */
public class CarriersStatementSupport  {
    private static final Logger logger = LoggerFactory.getLogger(CarriersStatementSupport.class);

    private MobileCarriersConfigService mobileCarriersConfigService;

    private Map<String, MobileCarriers> carriersStatementsMap = new HashMap<String, MobileCarriers>();
    private int maxStatementLength;
    private AtomicBoolean locked = new AtomicBoolean(false);

    public CarriersStatementSupport(MobileCarriersConfigService mobileCarriersConfigService) {
        this.mobileCarriersConfigService = mobileCarriersConfigService;
        reloadConfig();
    }

    public CarriersStatementSupport() {
        this(new MobileCarriersPropertiesConfigService());
    }

    public Optional<MobileCarriers> findCarriersByMobile(String mobile) {
        if (Strings.isNullOrEmpty(mobile)) {
            return Optional.absent();
        }
        // check lock
        while (locked.get()) {
            logger.warn("Carriers map is locked, so waiting for unlock.");
        }

        int statementLength = maxStatementLength;
        while (statementLength > 0) {
            // match statement
            String statement = mobile.substring(0, statementLength);
            MobileCarriers smsCarriers = carriersStatementsMap.get(statement);
            if (smsCarriers != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Found MobileCarriers: {} by mobile: {}", smsCarriers, mobile);
                }
                return Optional.of(smsCarriers);
            }
            statementLength--;
        }
        logger.warn("Could'nt found carriers by mobile: {}", mobile);
        return Optional.absent();
    }

    public boolean reloadConfig() {
        while (!locked.compareAndSet(false, true)) {
            logger.warn("Another thread is locked, so waiting for unlock.");
        }
        try {
            // clear cache
            carriersStatementsMap.clear();
            // load configs
            String mobileStatementConfig = mobileCarriersConfigService.getCarriersMobileStatementConfig();
            String unicomStatementConfig = mobileCarriersConfigService.getCarriersUnicomStatementConfig();
            String telecomStatementConfig = mobileCarriersConfigService.getCarriersTelecomStatementConfig();

            // load statement and carriers map
            boolean mobileOk = loadMobileStatement(MobileCarriers.MOBILE, mobileStatementConfig);
            boolean unicomOk = loadMobileStatement(MobileCarriers.UNICOM, unicomStatementConfig);
            boolean telecomOk = loadMobileStatement(MobileCarriers.TELECOM, telecomStatementConfig);

            // load max statement length
            loadMaxStatementLength();
            return mobileOk || unicomOk || telecomOk;
        } catch (Exception e) {
            logger.error("", e);
            return false;
        } finally {
            locked.compareAndSet(true, false);
        }
    }

    private void loadMaxStatementLength() {
        maxStatementLength = -1;
        Set<String> keySet = carriersStatementsMap.keySet();
        for (String statement : keySet) {
            int length = statement.length();
            if (length > maxStatementLength) {
                maxStatementLength = length;
            }
        }
    }

    private boolean loadMobileStatement(MobileCarriers carriers, String statementConfig) {
        Optional<List<String>> optional = parseStatementConfig(statementConfig);
        if (!optional.isPresent()) {
            logger.error("No mobile carriers statement found!");
            return false;
        } else {
            List<String> list = optional.get();
            for (String statement : list) {
                carriersStatementsMap.put(statement, carriers);
            }
            return true;
        }
    }


    private Optional<List<String>> parseStatementConfig(String statementConfig) {
        if (Strings.isNullOrEmpty(statementConfig)) {
            return Optional.absent();
        }
        List<String> result = new ArrayList<String>();
        String[] statements = statementConfig.split(CarriersUtils.CARRIERS_DELIMITER);
        for (String statement : statements) {
            result.add(statement.trim());
        }
        return Optional.of(result);
    }

}
