package com.xunlei.channel.common.utils.scan;


import com.xunlei.channel.common.utils.Assert;
import com.xunlei.channel.common.utils.CollectionUtils;
import com.xunlei.channel.common.utils.string.StringUtils;

import java.io.File;
import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * @author xiongyingqi
 * @since 20171019//
 */
public class PackageScanner {
    private Collection<Package> packages;
    private Collection<Class<? extends Annotation>> orAnnotations;
    private Collection<Class<? extends Annotation>> andAnnotations;
    private Collection<String> startWithStrings;
    private Collection<String> endWithStrings;
    private Collection<Class<?>> orImplementsInterface;
    private Collection<Class<?>> andImplementsInterface;
    private boolean recursive;
    private boolean ignoreInterface;

    private PackageScanner() {
    }

    public static PackageScanner newScanner() {
        PackageScanner packageScanner = new PackageScanner();
        packageScanner.packages = new ArrayList<Package>();
        packageScanner.orAnnotations = new ArrayList<Class<? extends Annotation>>();
        packageScanner.andAnnotations = new ArrayList<Class<? extends Annotation>>();
        packageScanner.startWithStrings = new ArrayList<String>();
        packageScanner.endWithStrings = new ArrayList<String>();
        packageScanner.orImplementsInterface = new ArrayList<Class<?>>();
        packageScanner.andImplementsInterface = new ArrayList<Class<?>>();

        packageScanner.recursive = true;
        packageScanner.ignoreInterface = false;
        return packageScanner;
    }

    /**
     * 增加扫描的包
     *
     * @param pkg
     * @return
     */
    public PackageScanner addPackage(String pkg) {
        Package aPackage = Package.getPackage(pkg);
        addPackage(aPackage);
        return this;
    }

    /**
     * 增加扫描的包
     *
     * @param pkg
     * @return
     */
    public PackageScanner addPackage(Package pkg) {
        packages.add(pkg);
        return this;
    }

    /**
     * 是否递归子包
     *
     * @param recursive
     */
    public PackageScanner recursive(boolean recursive) {
        this.recursive = recursive;
        return this;
    }

    public PackageScanner ignoreInterface(boolean ignore) {
        this.ignoreInterface = ignore;
        return this;
    }

    /**
     * 或者类被annotation注解
     *
     * @return
     */
    public PackageScanner orAnnotation(Class<? extends Annotation> annotation) {
        orAnnotations.add(annotation);
        return this;
    }

    /**
     * 并且类被annotation注解
     *
     * @param annotation
     * @return
     */
    public PackageScanner andAnnotation(Class<? extends Annotation> annotation) {
        andAnnotations.add(annotation);
        return this;
    }

    /**
     * 并且类实现接口
     *
     * @param interfc
     * @return
     */
    public PackageScanner andInterface(Class<?> interfc) {
        andImplementsInterface.add(interfc);
        return this;
    }

    /**
     * 或者类实现接口
     *
     * @param interfc
     * @return
     */
    public PackageScanner orInterface(Class<?> interfc) {
        orImplementsInterface.add(interfc);
        return this;
    }

    /**
     * @param startWithString
     * @return
     */
    public PackageScanner orStartWithString(String startWithString) {
        if (StringUtils.isEmpty(startWithString)) {
            return this;
        }
        startWithStrings.add(startWithString);
        return this;
    }

    public PackageScanner orEndWithString(String endWithString) {
        if (StringUtils.isEmpty(endWithString)) {
            return this;
        }
        endWithStrings.add(endWithString);
        return this;
    }


    /**
     * 开始扫描
     */
    public Collection<Class<?>> scan() {
        Assert.notEmpty(packages, "待扫描的包为空！");
        Collection<Class<?>> classes = new LinkedHashSet<Class<?>>();
        ClassLookupHelper.ClassFileFilter classFileFilter = new ClassLookupHelper.ClassFileFilter() {
            @Override
            public boolean accept(String klassName, File file, ClassLoader loader) {
                return true;
            }

            @Override
            public boolean accept(String klassName, JarFile jar, JarEntry entry, ClassLoader loader) {
                return true;
            }
        };
        for (Package pkg : packages) {
            Set<Class<?>> scanClasses = ClassLookupHelper.getClasses(pkg, recursive, classFileFilter);
            for (Class<?> scanClass : scanClasses) {
                if (isAnnotationPassed(scanClass) && isStartWithPassed(scanClass) && isInterfacePassed(scanClass)) {
                    classes.add(scanClass);
                }
            }
        }
        return classes;
    }

    private boolean isStartWithPassed(Class<?> clazz) {
        // 如果是需要判断开始和结尾，则初始值为false
        if (CollectionUtils.isEmpty(startWithStrings) && CollectionUtils.isEmpty(endWithStrings)) {// 如果需要判断的为空，则通过
            return true;
        }

        for (String startWithString : startWithStrings) {
            if (clazz.getSimpleName().startsWith(startWithString)) {
                return true;
            }
        }

        for (String endWithString : endWithStrings) {
            if (clazz.getSimpleName().endsWith(endWithString)) {
                return true;
            }
        }

        return false;
    }

    private boolean isAnnotationPassed(Class<?> clazz) {
        //true ^ true = false, true ^ false = true
        boolean flag = CollectionUtils.isEmpty(orAnnotations) && CollectionUtils.isEmpty(andAnnotations);// 如果判断是否需要注解，则初始值为false

        for (Class<? extends Annotation> orAnnotation : orAnnotations) {
            boolean annotationPresent = clazz.isAnnotationPresent(orAnnotation);
            if (annotationPresent) {
                flag = true;
                break;
            }
        }


        for (Class<? extends Annotation> andAnnotation : andAnnotations) {
            boolean annotationPresent = clazz.isAnnotationPresent(andAnnotation);
            if (!annotationPresent) {
                flag = false;
                break;
            }
        }
        return flag;
    }

    private boolean isInterfacePassed(Class<?> clazz) {
        //true ^ true = false, true ^ false = true
        boolean flag = CollectionUtils.isEmpty(orImplementsInterface) && CollectionUtils.isEmpty(andImplementsInterface);// 如果判断是否需要注解，则初始值为false
        if (ignoreInterface && clazz.isInterface()) {
            return false;
        }
        for (Class<?> orInterface : orImplementsInterface) {
            boolean isImplementsInterface = orInterface.isAssignableFrom(clazz);
//            EntityHelper.print("clazz: " + clazz + ", isImplementsInterface: " + isImplementsInterface);
            if (isImplementsInterface) {
                flag = true;
                break;
            }
        }


        for (Class<?> andInterface : andImplementsInterface) {
            boolean isImplementsInterface = andInterface.isAssignableFrom(clazz);
            if (!isImplementsInterface) {
                flag = false;
                break;
            }
            flag = true;
        }
        return flag;
    }


    public static void main(String[] args) {
        Collection<Class<?>> scan = PackageScanner.newScanner().addPackage(PackageScanner.class.getPackage()).orAnnotation(Deprecated.class).scan();
        System.out.println(scan);
        Collection<Class<?>> scan2 = PackageScanner.newScanner().addPackage(ArrayList.class.getPackage()).andInterface(Collection.class).scan();
        System.out.println(scan2);
    }

}
