package com.xunlei.channel.common.utils.reflect;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;

/**
 * @author xiongyingqi
 * @since 16-11-30 下午7:38
 */
public abstract class ReflectionUtils {
    private static final Logger logger = LoggerFactory.getLogger(ReflectionUtils.class);

    public static <T> void setFieldValueToInstance(String name, String value, T instance, Field[] inFields) throws IllegalAccessException {
        Field field = null;

        for (Field inField : inFields) {
            if (inField.getName().equals(name)) {
                field = inField;
            }
        }

        if (field == null) {
            return;
        }
        field.setAccessible(true);
        field.set(instance, value);
    }

    public static Map<String, String> convertObjectFieldsToMap(Object o) {
        if (o == null) {
            return null;
        }

        return convertObjectFieldsToMap(o, o.getClass());
    }

    public static Map<String, String> convertObjectFieldsToMap(Object o, Class clazz) {
        return convertObjectFieldsToMap(o, clazz, true);
    }

    public static Map<String, String> convertObjectFieldsToMap(Object o, Class clazz, boolean recursion) {
        if (o == null) {
            return null;
        }
        Map<String, String> map = new HashMap<String, String>();

        Field[] fields = findFields(clazz, recursion);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                continue;
            }
            field.setAccessible(true);
            try {
                Object fieldValue = field.get(o);
                String name = field.getName();
                String fieldValueString = null;
                if (fieldValue != null) {
                    fieldValueString = fieldValue.toString();
                }

                map.put(name, fieldValueString);
            } catch (IllegalAccessException e) {
                logger.error("", e);
            }
        }

        if (logger.isDebugEnabled()) {
            logger.debug("Convert object: {} to map: {}", o, map);
        }

        return map;
    }

    public static List<String> findFieldNames(Class clazz, boolean recursion) {
        Field[] fields = findFields(clazz, recursion);
        List<String> fieldNames = new ArrayList<String>();
        for (Field field : fields) {
            fieldNames.add(field.getName());
        }
        return fieldNames;
    }

    public static Field[] findFields(Class clazz, boolean recursion) {
        List<Field> fields = new ArrayList<Field>();

        Field[] clazzDeclaredFields = clazz.getDeclaredFields();

        for (Field field : clazzDeclaredFields) {
            if (Modifier.isStatic(field.getModifiers())) {
                continue;
            }
            fields.add(field);
        }

        if (recursion) {
            for (Class superclass = clazz.getSuperclass(); superclass != Object.class; superclass = superclass.getSuperclass()) {
                Field[] superFields = findFields(superclass, false);
                fields.addAll(Arrays.asList(superFields));
            }
        }
        return fields.toArray(new Field[fields.size()]);
    }

    public static Method findGetMethod(Field field) throws NoSuchMethodException {
        if (field == null) {
            return null;
        }
        Class<?> declaringClass = field.getDeclaringClass();

        String name = field.getName();
        String suffix = name.substring(0, 1).toUpperCase();
        if (name.length() > 1) {
            suffix += name.substring(1);
        }
        String methodName = "get" + suffix;
        try {
            return declaringClass.getDeclaredMethod(methodName);
        } catch (NoSuchMethodException e) {
            if (field.getType() == Boolean.TYPE) {
                if (name.startsWith("is")) {
                    return declaringClass.getDeclaredMethod(name);
                }
                methodName = "is" + suffix;
                return declaringClass.getDeclaredMethod(methodName);
            }
            throw e;
        }
    }

    public static Object getFieldValue(Field field, Object instance) throws IllegalAccessException {
        field.setAccessible(true);
        return field.get(instance);
    }

}
