/**
 * Project: dbservice
 * File Created at 2014-12-2
 * Copyright 2014 XunLei.com Corporation Limited.
 * All rights reserved.
 * This software is the confidential and proprietary information of
 * XunLei Company. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with XunLei.com.
 */
package com.xunlei.channel.db.dao.impl;

import com.xunlei.channel.db.common.Constants;
import com.xunlei.channel.db.dao.PayOrderOkDAO;
import com.xunlei.channel.db.orm.PayOrderOkMapper;
import com.xunlei.channel.db.pojo.PayOrderOk;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.Date;
import java.util.List;

@Service
public class PayOrderOkDAOImpl implements PayOrderOkDAO {

    private final static Logger logger = LoggerFactory.getLogger(PayOrderOkDAOImpl.class);

    @Autowired
    private PayOrderOkMapper payOrderOkMapper;

    @Transactional(readOnly = false)
    @Override
    public void savePayOrderOk(PayOrderOk payOrderOk) throws DataAccessException {
        payOrderOkMapper.savePayOrderOk(payOrderOk);
    }

    @Override
    public PayOrderOk getPayOrderOk(String xunleiPayId) throws DataAccessException {
        Assert.notNull(xunleiPayId);
        return payOrderOkMapper.getPayOrderOk(xunleiPayId);
    }

    @Transactional(readOnly = false)
    @Override
    public void updatePayOrderOk(PayOrderOk payOrderOk) throws DataAccessException {
        Assert.notNull(payOrderOk);
        logger.debug("save payOrderOk...xunleiPayId:{}", payOrderOk.getXunleiPayId());
        payOrderOkMapper.updatePayOrderOk(payOrderOk);
    }

    @Override
    public PayOrderOk getPayOrderOkByBizNoAndBizOrderId(String bizNo, String bizOrderId)
            throws DataAccessException {
        Assert.notNull(bizNo);
        Assert.notNull(bizOrderId);
        return payOrderOkMapper.getPayOrderOkByBizNoAndBizOrderId(bizNo, bizOrderId);
    }

    @Override
    public PayOrderOk getPayOrderOkByBizOrderId(String bizOrderId) {
        Assert.notNull(bizOrderId);
        return payOrderOkMapper.getPayOrderOkByBizOrderId(bizOrderId);
    }

    /*
     * (non-Javadoc)
     * @see
     * com.xunlei.channel.db.dao.PayOrderOkDAO#getPayOrderOkSendNoticeStatus
     * (java.lang.String)
     */
    @Override
    public String getPayOrderOkSendNoticeStatus(String xunleiPayId) throws DataAccessException {
        Assert.notNull(xunleiPayId);
        return this.payOrderOkMapper.getPayOrderOkSendNoticeStatus(xunleiPayId);
    }

    /*
     * (non-Javadoc)
     * @see
     * com.xunlei.channel.db.dao.PayOrderOkDAO#updateSendNoticeStatus(java.lang
     * .String, java.lang.String)
     */
    @Transactional(readOnly = false)
    @Override
    public void updateSendNoticeStatus(String xunleiPayId, String sendNoticeStatus)
            throws DataAccessException {
        Assert.notNull(xunleiPayId);
        Assert.notNull(sendNoticeStatus);
        this.payOrderOkMapper.updateSendNoticeStatus(xunleiPayId, sendNoticeStatus);
    }

    @Override
    public List<String> listUnsuccessNoticeXunleiPayId(Date fromTime, Date endTime)
            throws DataAccessException {
        if (null == fromTime && null == endTime) {
            return null;
        }
        return this.payOrderOkMapper.listUnsuccessNoticeXunleiPayId(
                Constants.SEND_NOTICE_STATUS_FAIL, fromTime, endTime);
    }

    /**
     * 根据手机号、业务号、开始日期、结束日期查询订单数量
     *
     * @param phone    手机号
     * @param bizNo    业务号
     * @param fromDate 开始日期
     * @param toDate   结束日期
     * @return 订单数
     * @throws DataAccessException
     */
    @Override
    public int getCountByPhoneBizNoAndDate(String phone, String bizNo, Date fromDate, Date toDate)
            throws DataAccessException {
        Assert.notNull(phone);
        Assert.notNull(bizNo);
        Assert.notNull(fromDate);
        Assert.notNull(toDate);
        return this.payOrderOkMapper.getCountByPhoneBizNoAndDate(phone, bizNo, fromDate, toDate);
    }

    /**
     * 获取用户一定日期内的成功订单总金额（必须以分为单位）
     *
     * @param xunleiId 迅雷id
     * @param bizNo    业务编号
     * @param fromDate 开始日期
     * @param toDate   结束日期
     * @return
     * @throws DataAccessException
     */
    @Override
    public int getSumPayAmountByXunleiIdBizNoAndDate(String xunleiId, String bizNo, Date fromDate,
                                                     Date toDate) throws DataAccessException {

        Integer sum = this.payOrderOkMapper.getSumPayAmountByXunleiIdBizNoAndDate(xunleiId, bizNo,
                fromDate, toDate);
        return sum == null ? 0 : sum;
    }

    /**
     * 帐号内一定时间内的成功次数
     *
     * @param xunleiId 迅雷Id
     * @param bizNo    业务编号
     * @param fromDate 开始日期
     * @param toDate   结束日期
     * @return
     * @throws DataAccessException
     */
    @Override
    public int getCountByXunleiIdBizNoAndDate(String xunleiId, String bizNo, Date fromDate,
                                              Date toDate) throws DataAccessException {
        return payOrderOkMapper.getCountByXunleiIdBizNoAndDate(xunleiId, bizNo, fromDate, toDate);
    }

    @Override
    public PayOrderOk getPayOrderOkByPayTypeAndChannelOrderId(String payType, String channelOrderId)
            throws DataAccessException {
        Assert.notNull(payType);
        Assert.hasText(channelOrderId, "channelOrderId could not be null!");
        return payOrderOkMapper.getPayOrderOkByPayTypeAndChannelOrderId(payType, channelOrderId);
    }

    @Override
    public PayOrderOk getFirstPayOrderOkByXunleiIdAndPayType(String xunleiId, String payType)
            throws DataAccessException {
        Assert.hasText(xunleiId, "xunleiId could not be null!");
        Assert.hasText(payType, "payType could not be null!");
        return payOrderOkMapper.getFirstPayOrderOkByXunleiIdAndPayType(xunleiId, payType);
    }

    @Override
    public PayOrderOk getLastPayOrderOkByXunleiIdAndPayType(String xunleiId, String payType)
            throws DataAccessException {
        Assert.hasText(xunleiId, "xunleiId could not be null!");
        Assert.hasText(payType, "payType could not be null!");
        return payOrderOkMapper.getLastPayOrderOkByXunleiIdAndPayType(xunleiId,
                payType);
    }

    @Override
    public long getSumOfOrderByXunleiIdAndDate(String xunleiId, Date fromDate, Date toDate) {
        Assert.notNull(xunleiId);
        Long sum = this.payOrderOkMapper.getSumOfOrderByXunleiIdAndDate(xunleiId, fromDate,
                toDate);
        return sum == null ? 0L : sum;
    }

    @Override
    public long getSumOfOrderByXunleiIdPayTypeAndDate(String xunleiId, String payType,
                                                      Date fromDate, Date toDate) {

        Assert.notNull(xunleiId);
        Long sum = this.payOrderOkMapper.getSumOfOrderByXunleiIdPayTypeAndDate(xunleiId, payType,
                fromDate,
                toDate);
        return sum == null ? 0L : sum;
    }
}
