package com.xunlei.channel.common.platform.spring.dao;

import com.xunlei.channel.common.utils.JacksonHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.PreparedStatementCreatorFactory;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.jdbc.support.GeneratedKeyHolder;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author xiongyingqi
 * @since 20171013//
 */
public abstract class BaseDao<T> {
    private static final Logger logger = LoggerFactory.getLogger(BaseDao.class);
    private Class<T> pojoType;

    @SuppressWarnings("unchecked")
    public BaseDao() {
        Type genericSuperclass = this.getClass().getGenericSuperclass();
        try {
            Type type = ((ParameterizedType) genericSuperclass).getActualTypeArguments()[0];
            pojoType = (Class<T>) type;
            logger.info("constructor... genericSuperclass: {}, pojoType: {}", genericSuperclass,
                    pojoType);
        } catch (ClassCastException e) {
            logger.error("subclass must preset a generic class!", e);
        }
    }

/*    protected Collection<T> query(String sql) {
        List<T> query = getJdbcTemplate().query(sql, new BeanPropertyRowMapper<T>(pojoType));
        if (logger.isDebugEnabled()) {
            logger.debug("query... sql: {}, result: {}", sql, JacksonHelper.getJsonString(query));
        }
        return query;
    }*/

    protected List<T> query(String sql, Object... args) {
        List<T> result = getJdbcTemplate().query(sql, new BeanPropertyRowMapper<T>(pojoType), args);
        if (logger.isDebugEnabled()) {
            logger.debug("query... sql: {}, args: {} result: {}", sql, args, JacksonHelper
                    .getJsonString(result));
        }
        return result;
    }

    protected List<T> queryByParameterType(String sql, Object parameterType) {
        try {
            SqlParameterSource parameterSource = new BeanPropertySqlParameterSource(parameterType);
            BeanPropertyRowMapper<T> beanPropertyRowMapper = new BeanPropertyRowMapper<T>(pojoType);
            List<T> result = getNamedParameterJdbcTemplate()
                    .query(sql, parameterSource, beanPropertyRowMapper);
            if (logger.isDebugEnabled()) {
                logger.debug("queryForObject... sql: {}, args: {}, result: {}", sql, parameterType,
                        JacksonHelper.getJsonString(result));
            }
            return result;
        } catch (EmptyResultDataAccessException e) {
            logger.error("", e);
            return new ArrayList<T>();
        }
    }

    protected T queryForObject(String sql, Object... args) {
        try {
            T result = getJdbcTemplate()
                    .queryForObject(sql, new BeanPropertyRowMapper<T>(pojoType), args);
            if (logger.isDebugEnabled()) {
                logger.debug("queryForObject... sql: {}, args: {}, result: {}", sql, args,
                        JacksonHelper.getJsonString(result));
            }
            return result;
        } catch (EmptyResultDataAccessException e) {
            logger.error("", e);
            return null;
        }
    }

    protected T queryForObject(String sql, T t) {
        try {
            SqlParameterSource parameterSource = new BeanPropertySqlParameterSource(t);
            BeanPropertyRowMapper<T> beanPropertyRowMapper = new BeanPropertyRowMapper<T>(pojoType);
            T result = getNamedParameterJdbcTemplate()
                    .queryForObject(sql, parameterSource, beanPropertyRowMapper);
            if (logger.isDebugEnabled()) {
                logger.debug("queryForObject... sql: {}, args: {}, result: {}", sql, t,
                        JacksonHelper.getJsonString(result));
            }
            return result;
        } catch (EmptyResultDataAccessException e) {
            logger.error("", e);
            ;
            return null;
        }
    }

    protected int update(String sql, Object... args) {
        int update = getJdbcTemplate().update(sql, args);
        if (logger.isDebugEnabled()) {
            logger.debug("update... sql: {}, args: {} result: {}", sql, args, update);
        }
        return update;
    }

    protected int update(String sql, T t) {
        SqlParameterSource parameterSource = new BeanPropertySqlParameterSource(t);
        int update = getNamedParameterJdbcTemplate().update(sql, parameterSource);
        if (logger.isDebugEnabled()) {
            logger.debug("update... sql: {}, args: {} result: {}", sql, t, update);
        }
        return update;
    }

    protected int[] batchUpdate(String sql, T[] t) {
        int length = t.length;
        SqlParameterSource[] parameterSources = new BeanPropertySqlParameterSource[length];
        for (int i = 0; i < t.length; i++) {
            parameterSources[i] = new BeanPropertySqlParameterSource(t[i]);
        }

        int[] updates = getNamedParameterJdbcTemplate().batchUpdate(sql, parameterSources);
        if (logger.isDebugEnabled()) {
            logger.debug("update... sql: {}, args: {} result: {}", sql, t, Arrays.toString(updates));
        }
        return updates;
    }

    protected int[] updateEntities(String sql, T[] array) {
        SqlParameterSource[] parameterSources = new BeanPropertySqlParameterSource[array.length];
        for (int i = 0; i < array.length; i++) {
            BeanPropertySqlParameterSource sqlParameterSource = new BeanPropertySqlParameterSource(
                    array[i]);
            parameterSources[i] = sqlParameterSource;
        }
        int[] update = getNamedParameterJdbcTemplate().batchUpdate(sql, parameterSources);
        if (logger.isDebugEnabled()) {
            logger.debug("update... sql: {}, args: {} result: {}", sql, Arrays.toString(array),
                    Arrays.toString(update));
        }
        return update;
    }

    protected GeneratedKeyHolder updateAndReturnGeneratedKeyHolder(String sql, int[] types,
                                                                   Object[] parameters) {
        PreparedStatementCreatorFactory factory = new PreparedStatementCreatorFactory(sql, types);
        PreparedStatementCreator preparedStatementCreator = factory
                .newPreparedStatementCreator(parameters);
        GeneratedKeyHolder generatedKeyHolder = new GeneratedKeyHolder();
        int update = getJdbcTemplate().update(preparedStatementCreator, generatedKeyHolder);
        if (logger.isDebugEnabled()) {
            logger.debug(
                    "update... sql: {}, types: {}, parameters: {}, updates: {}, generatedKeyHolder: {}",
                    sql,
                    Arrays.toString(types), Arrays.toString(parameters), update,
                    generatedKeyHolder);
        }
        return generatedKeyHolder;
    }

    protected GeneratedKeyHolder updateAndReturnGeneratedKeyHolder(String sql, T entity) {
        SqlParameterSource parameterSource = new BeanPropertySqlParameterSource(entity);
        GeneratedKeyHolder generatedKeyHolder = new GeneratedKeyHolder();
        int update = getNamedParameterJdbcTemplate().update(sql, parameterSource, generatedKeyHolder);
        if (logger.isDebugEnabled()) {
            logger.debug(
                    "update... sql: {}, parameters: {}, updates: {}, generatedKeyHolder: {}",
                    sql, entity, update, generatedKeyHolder);
        }
        return generatedKeyHolder;
    }


    abstract JdbcTemplate getJdbcTemplate();

    abstract NamedParameterJdbcTemplate getNamedParameterJdbcTemplate();
}
