package com.xunlei.channel.common.platform.spring.aop;

import com.google.common.base.Optional;
import com.xunlei.channel.common.platform.entity.BizRequestBase;
import com.xunlei.channel.common.platform.entity.ReturnResult;
import com.xunlei.channel.common.platform.service.SignService;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import java.util.Arrays;

/**
 * @author xiongyingqi
 * @since 20171012//
 */
@Aspect
@Component
@Order(1)
public class CheckSignAspect {
    private static final Logger logger = LoggerFactory.getLogger(CheckSignAspect.class);
    private SignService signService;

    @Pointcut("execution(public com.xunlei.channel.common.platform.entity.ReturnResult+ (com.xunlei..*)" +
            ".*(.., @com.xunlei.channel.common.platform.spring.annotation.CheckSign (com.xunlei.channel.common.platform.entity.BizRequestBase+), ..))")
    public void checkSignMethods() {
    }

    @Around("checkSignMethods()")
    public Object aop(ProceedingJoinPoint pjp) throws Throwable {
        Object[] args = pjp.getArgs();
        for (Object arg : args) {
            try {
                Optional<ReturnResult> returnResultOptional = checkSign(arg);
                if (returnResultOptional.isPresent() && !returnResultOptional.get().isSuccess()) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Intercept check sign result, and returns: {}", returnResultOptional.get());
                    }
                    return returnResultOptional.get();
                }
            } catch (Exception e) {
                logger.error("Error when check sign of arg: " + arg + ". Message: " + e.getMessage(), e);
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Passed check sign. args: {}", Arrays.toString(args));
        }
        return pjp.proceed();
    }

    private Optional<ReturnResult> checkSign(Object arg) {
        if (!(arg instanceof BizRequestBase)) {
            return Optional.absent();
        }
        BizRequestBase requestBase = (BizRequestBase) arg;

        Boolean signOk = signService.checkSign(requestBase);
        if (!signOk) {
            logger.warn("Failed match sign! request: {}", requestBase);
            return Optional.of(ReturnResult.CHECK_SIGN_ERROR);
        }
        return Optional.absent();
    }

    public SignService getSignService() {
        return signService;
    }

    public void setSignService(SignService signService) {
        this.signService = signService;
    }
}
