package com.xunlei.channel.common.platform.spring.aop;

import com.google.common.base.Optional;
import com.xunlei.channel.common.platform.entity.ReturnResult;
import com.xunlei.channel.common.platform.util.BindingResultUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindingResult;

/**
 * @author xiongyingqi
 * @since 16-12-15 下午9:13
 */
@Aspect
@Component
@Order(0)
public class BindingResultAspect {
    private static final Logger logger = LoggerFactory.getLogger(BindingResultAspect.class);

    @Pointcut("execution(public com.xunlei.channel.common.platform.entity.ReturnResult+ (com.xunlei.channel..*)" +
            ".*(.., @javax.validation.Valid (*),org.springframework.validation.BindingResult, ..))")
    public void validMethods() {
    }

    @Around("validMethods()")
    public Object aop(ProceedingJoinPoint pjp) throws Throwable {
        Object[] args = pjp.getArgs();
        for (Object arg : args) {
            Optional<ReturnResult> returnResultOptional = buildResultIfHasErrors(arg);
            if (returnResultOptional.isPresent()) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Intercept binding result, and returns: {}", returnResultOptional.get());
                }
                return returnResultOptional.get();
            }
        }
        return pjp.proceed();
    }

    private Optional<ReturnResult> buildResultIfHasErrors(Object arg) {
        if (!(arg instanceof BindingResult)) {
            return Optional.absent();
        }
        BindingResult bindingResult = (BindingResult) arg;
        if (!bindingResult.hasErrors()) {
            return Optional.absent();
        }
        ReturnResult returnResult = BindingResultUtils.buildResult(bindingResult);
        if (returnResult == null) {
            return Optional.absent();
        }
        return Optional.of(returnResult);
    }


}
