package com.xunlei.channel.common.logic.handle.impl;

import com.xunlei.channel.common.logic.config.constants.FieldQualified;
import com.xunlei.channel.common.logic.config.constants.LogicalOperation;
import com.xunlei.channel.common.logic.config.method.ConvertException;
import com.xunlei.channel.common.logic.config.method.ConvertMethod;
import com.xunlei.channel.common.logic.config.method.MethodManager;
import com.xunlei.channel.common.logic.config.vo.Condition;
import com.xunlei.channel.common.logic.config.vo.Conditions;
import com.xunlei.channel.common.logic.handle.ConditionFieldHandler;
import com.xunlei.channel.common.logic.handle.HandlerException;
import com.xunlei.channel.common.logic.handle.LogicHandler;
import com.xunlei.channel.common.logic.handle.LogicOperation;
import com.xunlei.channel.common.logic.vo.InputData;
import com.xunlei.channel.common.utils.Assert;
import com.xunlei.channel.common.utils.CollectionUtils;
import com.xunlei.channel.common.utils.string.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 逻辑处理器，根据配置、输入对象、方法管理器进行逻辑处理
 *
 * @author xiongyingqi
 * @since 20171019//
 */
public class LogicHandlerImpl<T> implements LogicHandler<T> {
    private static final Logger logger = LoggerFactory.getLogger(LogicHandlerImpl.class);

    @Override
    public boolean handle(T inputData, Conditions conditions, MethodManager methodManager) throws HandlerException {
        Assert.notNull(inputData, "input logicData is null!");
        Assert.notNull(methodManager, "methodManager is null!");
        Map<String, String> fieldNameAndValueMap = ConditionFieldHandler.processFieldAndValue(inputData);
        Assert.notEmpty(fieldNameAndValueMap, "Could'nt get field map of input logicData: " + inputData);
        return handleConditions(conditions, fieldNameAndValueMap, methodManager);
    }

    protected boolean handleConditions(Conditions conditions, Map<String, String> fieldNameAndValueMap, MethodManager methodManager) throws HandlerException {
        if (conditions == null) {
            return true;
        }
        if (!conditions.isAvailable()) {
            return false;
        }
        // result of subConditions
        Conditions subConditions = conditions.getSubConditions();
        boolean flagOfSubConditions = handleConditions(subConditions, fieldNameAndValueMap, methodManager);

        // result of
        List<Condition> conditionsCollection = conditions.getConditions();
        boolean flagOfConditionCollection = handleConditions(conditionsCollection, fieldNameAndValueMap, methodManager);

        LogicalOperation logicalOperation = conditions.getLogicalOperation();
        if (logicalOperation == null) {
            logger.error("Null of logicalOperation on config: {} input data: {}", conditions, fieldNameAndValueMap);
            throw new HandlerException("Null of logicalOperation on config: " + conditions);
        }
        boolean result = LogicOperation.execute(logicalOperation, flagOfSubConditions, flagOfConditionCollection);
        if (logger.isDebugEnabled()) {
            logger.debug("Return result: {} of conditions: {} fieldMap: {} methodManager: {}", result, conditions, fieldNameAndValueMap, methodManager);
        }
        return result;
    }


    protected boolean handleConditions(Collection<Condition> conditionCollection, Map<String, String> fieldNameAndValueMap, MethodManager methodManager) throws HandlerException {
        if (CollectionUtils.isEmpty(conditionCollection)) {
            return true;
        }
        for (Condition condition : conditionCollection) {
            boolean b = handleCondition(condition, fieldNameAndValueMap, methodManager);
            if (!b) {
                return false;
            }
        }
        return true;
    }

    protected boolean handleCondition(Condition condition, Map<String, String> fieldNameAndValueMap, MethodManager methodManager) throws HandlerException {
        if (!condition.isAvailable()) {
            return true;
        }
        String configValue = condition.getValue();
        String fieldValue = handleFieldValue(condition, fieldNameAndValueMap, methodManager);
        if (StringUtils.isEmpty(fieldValue) || StringUtils.isEmpty(configValue)) {
            return false;
        }
        FieldQualified qualified = condition.getQualified();
        boolean b = handleFieldQualified(fieldValue, configValue, qualified);
        if (logger.isDebugEnabled()) {
            logger.debug("Returns result: {} of configValue: {} fieldValue: {} qualified: {}", b, configValue, fieldValue, qualified);
        }
        return b;

    }

    protected boolean handleFieldQualified(String fieldValue, String configValue, FieldQualified qualified) throws HandlerException {
        switch (qualified) {
            case EQUALS:
                return fieldValue.equals(configValue);
            case IN:
                Collection<String> configs = Condition.splitConfigValue(configValue);
                return configs.contains(fieldValue);
            case CONTAINS:
                return fieldValue.contains(configValue);
            default:
                logger.error("Unknown qualified: {}", qualified);
                throw new HandlerException("Unknown qualified: " + qualified);
        }
    }


    protected String handleFieldValue(Condition condition, Map<String, String> fieldNameAndValueMap, MethodManager methodManager) {
        String name = condition.getName();
        if (StringUtils.isEmpty(name)) {
            return null;
        }
        String fieldValue = fieldNameAndValueMap.get(name);

        String valueConvertMethod = condition.getValueConvertMethod();
        if (valueConvertMethod != null) {
            ConvertMethod method = methodManager.getMethod(valueConvertMethod);
            try {
                return method.execute(fieldValue, condition);
            } catch (ConvertException e) {
                logger.error("Error when invoke method: " + method + " on condition: " + condition + ". ERROR: " + e.getMessage(), e);
            }
        }
        return fieldValue;
    }


}
