package com.xunlei.channel.common.logic.config.method;

import com.xunlei.channel.common.logic.config.annotation.ConvertMethodName;
import com.xunlei.channel.common.utils.scan.PackageScanner;
import com.xunlei.channel.common.utils.string.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * @author xiongyingqi
 * @since 20171019//
 */
public class MethodManager {
    private static final Logger logger = LoggerFactory.getLogger(MethodManager.class);

    private Map<String, ConvertMethod> methodNameAndInstanceMap = new HashMap<String, ConvertMethod>();
    public static final Package DEFAULT_PACKAGE = MethodManager.class.getPackage();

    private MethodManager(Package... pkg) {
        init(pkg);
    }

    @SuppressWarnings("unchecked")
    private void init(Package... pkg) {
        PackageScanner packageScanner = PackageScanner
                .newScanner()
                .addPackage(DEFAULT_PACKAGE)
                .andInterface(ConvertMethod.class)
                .ignoreInterface(true);
        for (Package aPackage : pkg) {
            packageScanner.addPackage(aPackage);
        }
        Collection<Class<?>> methodClasses = packageScanner.scan();
        logger.info("Found method classes: {}", methodClasses);
        for (Class<?> methodClass : methodClasses) {
            Class<ConvertMethod> convertMethodClass = (Class<ConvertMethod>) methodClass;
            String methodName = getMethodName(convertMethodClass);
            try {
                ConvertMethod convertMethod = convertMethodClass.newInstance();
                ConvertMethod exists = methodNameAndInstanceMap.put(methodName, convertMethod);
                if (exists != null) {
                    logger.error("Already exists instance: {} of name: {} new: {}", exists, methodName, convertMethod);
                } else {
                    logger.info("Put instance: {} of name: {}", convertMethod, methodName);
                }
            } catch (Exception e) {
                logger.error("Error when instance class: " + convertMethodClass + ", Error: " + e.getMessage(), e);
            }
        }
    }

    public ConvertMethod getMethod(String methodName) {
        return methodNameAndInstanceMap.get(methodName);
    }


    public static MethodManager newInstance() {
        return new MethodManager(DEFAULT_PACKAGE);
    }

    public static MethodManager newInstance(Package... packages) {
        return new MethodManager(packages);
    }

    private String getMethodName(Class<ConvertMethod> convertMethodClass) {
        ConvertMethodName annotation = convertMethodClass.getAnnotation(ConvertMethodName.class);
        String value = annotation.value();
        if (StringUtils.isEmpty(value)) {
            String simpleName = convertMethodClass.getSimpleName();
            return simpleName.substring(0, 1).toLowerCase() + simpleName.substring(1);
        }
        return value;
    }

}
