package com.xunlei.channel.common.logic.handle;

import com.xunlei.channel.common.logic.config.annotation.ConditionField;
import com.xunlei.channel.common.logic.config.annotation.ConditionFieldIgnore;
import com.xunlei.channel.common.logic.handle.constants.ConditionFieldType;
import com.xunlei.channel.common.utils.reflect.ReflectionUtils;
import com.xunlei.channel.common.utils.string.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

/**
 * 处理输入对象为map，形式是 fieldName -> fieldValue
 * @author xiongyingqi
 * @since 20171019//
 */
public class ConditionFieldHandler {
    private static final Logger logger = LoggerFactory.getLogger(ConditionFieldHandler.class);
    private static final Map<Class<?>, Collection<ConditionFieldEntry>> CACHE = new WeakHashMap<Class<?>, Collection<ConditionFieldEntry>>();

    /**
     * 处理输入对象为map，形式是 fieldName -> fieldValue
     * @param object 输入对象
     * @return name -> fieldValue
     * @throws HandlerException 处理异常
     */
    public static Map<String, String> processFieldAndValue(Object object) throws HandlerException {
        if (object == null) {
            return null;
        }
        Collection<ConditionFieldEntry> annotation = getAnnotation(object);
        HashMap<String, String> map = new HashMap<String, String>(annotation.size());
        for (ConditionFieldEntry conditionFieldEntry : annotation) {
            processEntry(object, conditionFieldEntry, map);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Process object: {} to field string: {}", object, map);
        }

        return map;
    }

    private static void processEntry(Object instance, ConditionFieldEntry entry, Map<String, String> toResult) throws HandlerException {
        ConditionFieldType conditionFieldType = entry.getConditionFieldType();
        String name = entry.getName();
        switch (conditionFieldType) {
            case FIELD:
                Field field = entry.getField();
                try {
                    Object fieldValue = ReflectionUtils.getFieldValue(field, instance);
                    if (fieldValue == null) {
                        return;
                    }
                    toResult.put(name, fieldValue.toString());
                } catch (IllegalAccessException e) {
                    logger.error("", e);
                }
                return;
            case METHOD:
                Method method = entry.getMethod();
                try {
                    Object value = method.invoke(instance);
                    if (value == null) {
                        return;
                    }
                    toResult.put(name, value.toString());
                } catch (Exception e) {
                    logger.error("Error when invoke method: " + method + ". ERROR: " + e.getMessage(), e);
                }
                return;
            default:
                throw new HandlerException("Unknown conditionFieldType: " + conditionFieldType);
        }
    }

    private static Collection<ConditionFieldEntry> getAnnotation(Object object) {
        Class<?> clazz = object.getClass();
        Collection<ConditionFieldEntry> exists = CACHE.get(object.getClass());
        if (exists != null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Returns conditionFieldEntry: {} by class: {}", exists, clazz);
            }
            return exists;
        }
        synchronized (ConditionFieldHandler.class) {
            Field[] fields = ReflectionUtils.findFields(clazz, true);
            ArrayList<ConditionFieldEntry> conditionFieldEntries = new ArrayList<ConditionFieldEntry>(fields.length);
            for (Field field : fields) {
                ConditionFieldEntry conditionFieldEntry = processField(field);
                if (conditionFieldEntry != null) {
                    conditionFieldEntries.add(conditionFieldEntry);
                }
            }
            logger.info("Initialized conditionFieldEntry: {} by class: {}", conditionFieldEntries, clazz);
            CACHE.put(clazz, conditionFieldEntries);
            return conditionFieldEntries;
        }
    }

    private static ConditionFieldEntry processField(Field field) {
        if (field.isAnnotationPresent(ConditionFieldIgnore.class)) {
            return null;
        }
        boolean annotationPresent = field.isAnnotationPresent(ConditionField.class);
        if (annotationPresent) {
            ConditionField annotation = field.getAnnotation(ConditionField.class);
            String value = annotation.value();
            if (StringUtils.isEmpty(value)) {
                value = field.getName();
            }
            return new ConditionFieldEntry(field, null, value, ConditionFieldType.FIELD);
        }
        Method getMethod = null;
        try {
            getMethod = ReflectionUtils.findGetMethod(field);
        } catch (NoSuchMethodException e) {
            logger.error("", e);
        }

        if (getMethod != null && getMethod.isAnnotationPresent(ConditionFieldIgnore.class)) {
            return null;
        }
        if (getMethod == null || !getMethod.isAnnotationPresent(ConditionField.class)) {
            return new ConditionFieldEntry(field, null, field.getName(), ConditionFieldType.FIELD);
        }
        ConditionField annotation = getMethod.getAnnotation(ConditionField.class);
        String value = annotation.value();
        if (StringUtils.hasLength(value)) {
            return new ConditionFieldEntry(field, getMethod, value, ConditionFieldType.METHOD);
        }
        return new ConditionFieldEntry(field, getMethod, field.getName(), ConditionFieldType.METHOD);
    }
}
