package com.firefly.iapp.security.web;

import com.firefly.iapp.security.CustomerAuthorizationManager;
import com.firefly.iapp.security.CustomerPrincipal;
import com.firefly.iapp.security.CustomerUserToken;
import com.firefly.iapp.security.CustomerUserTokenValue;
import lombok.extern.slf4j.Slf4j;
import org.beast.security.core.UserToken;
import org.beast.security.core.auth.UnauthenticatedException;
import org.beast.security.core.codec.UserTokenCodec;
import org.beast.security.core.exception.TokenException;
import org.beast.security.web.resolver.AbstractTokenArgumentResolver;
import org.beast.security.web.resolver.HeaderTokenValueWebExtractor;
import org.beast.web.util.WebUtils;
import org.springframework.core.MethodParameter;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.context.request.NativeWebRequest;

import java.util.Optional;

@Slf4j
public class CustomerUserTokenResolver extends AbstractTokenArgumentResolver<CustomerUserTokenValue> {
    private final UserTokenCodec codec = new UserTokenCodec();
    private HeaderTokenValueWebExtractor headerTokenValueWebExtractor;
    private CustomerAuthorizationManager authorizationManager;

    public CustomerUserTokenResolver(CustomerAuthorizationManager authorizationManager) {
        super(CustomerUserTokenValue.class);
        this.authorizationManager = authorizationManager;
        this.headerTokenValueWebExtractor = new HeaderTokenValueWebExtractor(WebUtils.HEADER_USER_TOKEN);
    }

    @Override
    public boolean isRequired(CustomerUserTokenValue annotation) {
        return annotation.required();
    }

    @Override
    protected void handleMissingValue(MethodParameter parameter) throws ServletRequestBindingException {
        throw new UnauthenticatedException("token missing");
    }

    @Override
    protected Object resolveValue(MethodParameter parameter, NativeWebRequest webRequest) throws ServletRequestBindingException {
        String tokenString = headerTokenValueWebExtractor.extract(parameter, webRequest);
        UserToken userToken;
        try {
            userToken = codec.decode(tokenString);
            userToken.verify();
        } catch (TokenException e) {
            log.warn("token exception", e);
            return null;
        } catch (Exception e) {
            log.warn("token invalid", e);
            return null;
        }
        String appId = userToken.getAppId();
        long uid = userToken.getUid();

        //确保是有效用户 之后再考虑优化
        CustomerPrincipal principal = authorizationManager.authorize(appId, uid);
        return CustomerUserToken.valueOf(userToken, principal.getId());

    }
}
