package com.firefly.design.util;

import com.firefly.design.si.Document;
import com.firefly.design.si.Element;
import com.firefly.design.si.ImageDocument;
import com.firefly.design.si.PageDocument;
import com.google.common.collect.Sets;
import org.springframework.util.ObjectUtils;

import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Predicate;

public class Documents {

    private static int MAX_DEPTH = 20;

    private static void walkEach(int depth, Iterable<Element> elements, Consumer<Element> consumer, Predicate<Element> preventDepth) {
        for (Element element : elements) {
            consumer.accept(element);
            var children = element.getChildren();
            if (depth < MAX_DEPTH && !ObjectUtils.isEmpty(children)) {
                if (preventDepth.test(element)) {
                    walkEach(depth + 1, children, consumer, preventDepth);
                }
            }
        }
    }
    public static void walk(Iterable<Element> elements, Consumer<Element> consumer, Predicate<Element> preventDepth) {
        walkEach(1, elements, consumer, preventDepth);
    }

    public static void walk(Iterable<Element> elements, Consumer<Element> consumer) {
        walkEach(1, elements, consumer, element -> true);
    }

    public static void walkElement(Document document, Consumer<Element> consumer) {
        if (document instanceof PageDocument pageDocument) {
            pageDocument.getLayouts().forEach(layout -> {
                walk(layout.getElements(), consumer);
            });
        } else if (document instanceof ImageDocument imageDocument) {
            walk(imageDocument.getElements(), consumer);
        }
    }

    public static Set<String> IMAGE_TAGS = Sets.newHashSet("Image", "ClippathImage");

    public static int countReplaceableImageCount(Iterable<Element> elements) {
        AtomicInteger count = new AtomicInteger();
        walk(elements, (element) -> {
            String as = element.getAs();
            if (IMAGE_TAGS.contains(as)) {
                var editable = element.getEditable();
                if (Objects.nonNull(editable) && Objects.equals(editable.get("replaceable"), Boolean.TRUE)) {
                    count.incrementAndGet();
                }
            }
        });

        return count.get();
    }
    public static int countReplaceableImageCount(Document document) {
        AtomicInteger count = new AtomicInteger();
        if (document instanceof PageDocument pageDocument) {
            pageDocument.getLayouts().forEach(layout -> {
                count.set(count.get() + countReplaceableImageCount(layout.getElements()));
            });
        } else if (document instanceof ImageDocument imageDocument) {
            count.set(countReplaceableImageCount(imageDocument.getElements()));;
        }
        return count.get();
    }

    public static int countLayouts(Document document) {
        if (document instanceof PageDocument pageDocument) {
            return pageDocument.getLayouts().size();
        }
        return 1;
    }


}
